/*----------------------------------------------------------------------------
 * Name     : bc5602.c
 * Purpose  : BC5602 API
 * Note(s)  : 
 *----------------------------------------------------------------------------
 *----------------------------------------------------------------------------*/
#include	"bc5602.h"
#include	"timer.h"

/*-------Macro ---------------------------------------------------------------*/
#define	delay_2cycle()		asm("jmp $+1")

#define	SPI_delay(varname)\
			asm("mov A,"#varname"");\
			asm("sdz acc");\
			asm("jmp $-1")
			
/*-------BC5602 I/O setting -----------------------------------------------*/
// IO1:b[7:6]
//   	0= 0.5mA
//   	1=   1mA
//   	2=   5mA
//   	3=  10mA
#define  _GPIO_CURRENT_		(1 << 6)

// IO3
//   b0	GPIO0 Pull-High Enable
//   b1	GPIO1 Pull-High Enable, SDI
//   b2	GPIO2 Pull-High Enable,	INT
//   b3	GPIO3 Pull-High Enable
//   b4	GPIO4 Pull-High Enable
//   b5	GPIO5 Pull-High Enable
//   b6	SPI CSN/SCLK/SDI Pull-High Enable
//   b7	SDO Tri-State Enable
#define  _SPI_PULL_HIGH_	(0xB9)

#define	_GIO1_FUNCTION_	SPI_SDO_FUN		/*GIO1=SPI MISO */
#define	_GIO2_FUNCTION_	INT_REQUEST		/*GIO2=IRQ */
#define	_GIO3_FUNCTION_	INPUT_MODE		/*GIO3=INPUT MODE */
#define	_GIO4_FUNCTION_	INPUT_MODE
/*-------BC5602 register parameter setting -------------------------------------------*/
#define	_XO_CAP_				0x15				/* Crystal CAP setting */
#define	AGC_ENABLE			_AGC_EN_

uc16 BC5602RegsSetup[]=
{
	B2W(CONFIG_REGS,REGS_BANK0),		// BANK0
	B2W(CLK_CTL2_REGS,0x00),			// RC2
	B2W(DLY_CTL_REGS,0x20),				// PKT5	
	B2W(MOD_CTL2_REGS,0x14),			// MOD2
	B2W(CP31_REGS,0x45),				// CP31
	B2W(XO_CAP_CTL_REGS,_XO_CAP_),		// XO1    
	B2W(XO_SEL_CTL_REGS,XCLK16M),		// XO2
	B2W(DMOD_CTL3_REGS,0x9A),			// DM3
	B2W(CONFIG_REGS,REGS_BANK1),		// BANK1
	B2W(AGC_CTL1_REGS,0x0C),			// AGC1
	B2W(AGC_CTL2_REGS,0x03),			// AGC2
	B2W(AGC_CTL4_REGS,0x10),			// AGC4
	B2W(FCF_CTL2_REGS,0xCC),			// FCF1
	B2W(FCF_CTL3_REGS,0x4C),			// FCF3
	B2W(FCF_CTL4_REGS,0x80),			// FCF4
	B2W(CONFIG_REGS,REGS_BANK2),		// BANK2
	B2W(0x21,0x77),						// RXG
	B2W(0x28,0xA0),    					// CP3[0x28]=0xA0: DLY_SYN=5:36us 
	B2W(0x2D,0x18),						// VC3
	B2W(0x2E,0xEC),						// RX1    
	B2W(0x36,0x03),    					// TX4[0x36]=0x03: DLY_PAD=40us 
	B2W(0x38,0x0A),						// CA2
	B2W(0x39,0x12),						// CA3
	B2W(0x3B,0x94),						// LD2    
	B2W(0x3C,0x43),						// LD3
	B2W(CONFIG_REGS,(AGC_ENABLE | REGS_BANK0)),	// BANK 0			
};

uc16	DataRateRegsSetup[]=	
{ B2W(MOD_CTL1_REGS,0x01),B2W(MOD_CTL1_REGS,0x03),B2W(MOD_CTL1_REGS,0x07) };

uc16	TxPowerRegsSetup[]=
{ 
	B2W(0x34,0xD2),B2W(0x35,0x2F),B2W(0x34,0x8B),B2W(0x35,0x01),
	B2W(0x34,0x8E),B2W(0x35,0x21),B2W(0x34,0xAF),B2W(0x35,0x21)
};
/*----------------------------------------------------------------------------*/
/* @brief 	: MCU & BC5602 interface Configure 											*
 *	@param	:																						*
 *	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void BC5602InterfaceConfigure(void)
{	
	/* setup MCU I/O for BC5602 interface */
	_cpc = 0x08;								/* C-/C+/CX for SPI I/O */	
	_pbpu5 = HIGH;								/* CSN pull-high */
	_pbpu6 = LOW;								/* SCK no pull-hihg */
	_papu1 = LOW;								/* SDO no pull-hihg */
	_papu3 = LOW;								/* SDI no pull-hihg */
	SPI_CSN_HIGH;								/* CSN=HIGH */
	/* setup IRQ interface 	*/
	_pbc0 = HIGH;								/* IRQ input mode */
	_pbpu0 = LOW;								/* IRQ no pull-high */
	/* IRQ falling edge */
	_int0s0 = LOW;
	_int0s1 = HIGH;							/* IRQ falling edge */
	
	_SPIC0_ = 0x00;							/* SPI master mode & SPI clock is Fsys/4 */
	_SPIC2_ = 0x38;							/* CKPOLB=1,CKEG=1,MLS=MSB first */
	_SPIEN_ = HIGH;							/* SPI enable */
	/* setup BC5602 SPI interface */
	BC5602LightSleepMode();		
   // Software reset BC5602
	BC5602SoftwareReset();
	Delay10us(1);
	BC5602RegisterBank(REGS_BANK0);	
	//change to 4-wire SPI	
	BC5602WriteRegister(GPIO12_CTL_REGS,_GPIO_CURRENT_ | _GIO1_FUNCTION_ | (_GIO2_FUNCTION_ << 3));
	BC5602WriteRegister(GPIO34_CTL_REGS,_GIO3_FUNCTION_ | (_GIO4_FUNCTION_ << 4));
	BC5602WriteRegister(GPIO_PULL_UP_REGS,_SPI_PULL_HIGH_);
}
/*----------------------------------------------------------------------------*/
/* @brief 	: BC5602 GPIO interface Configure 											
 	@param	:
					gpio - configure GPIO
					fun  - GPIO function
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void BC5602GPIOConfigure(u8 gpio,u8 fun)
{
   u8 reg,msk;
      
   if(gpio & 0x02)			/* GPIO3 or GPIO4 */
   {
  		reg = GPIO34_CTL_REGS;   	
   	fun &= 0x0F;
   	msk = 0xF0;
   	if(gpio & 0x01)		/* GPIO4 ? */
   	{
   		GCC_SWAP(msk);
   		GCC_SWAP(fun);
   	}
   }
   else
   {
  		reg = GPIO12_CTL_REGS;
   	fun &= 0x07;
   	msk = 0xF8;
   	if(gpio & 0x01)
   	{
   		msk = 0xC7;
   		GCC_SWAP(fun);
   		GCC_RR(fun);
   	}
   }
   BC5602ReadRegister(reg);
   GCC_ANDA(msk);
   GCC_ORM(fun);
   BC5602WriteRegister(reg,fun);
}
/*----------------------------------------------------------------------------*/
/* @brief 	: get IRQ line status.
 	@param	: 
 	@retval  :	FALSE or TRUE																	*/
/*----------------------------------------------------------------------------*/
u8 BC5602GetIRQLineStatus(void)
{
	return((IRQ_STATUS)?FALSE:TRUE);
}
/*----------------------------------------------------------------------------*/
/* @brief 	: get POWER ON status.
 	@param	: 
 	@retval  :	FALSE or TRUE																	*/
/*----------------------------------------------------------------------------*/
u8 BC5602GetPWRONStatus(void)
{
	if(BC5602ReadRegister(CLK_CTL1_REGS) & _PWRON_) return(TRUE);	
	return(FALSE);
}
/*----------------------------------------------------------------------------*/
/* @brief 	: BC5602 register set to default value
 	@param	: 
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void BC5602RegisterConfigure(void)
{
	u8 x;
	
	x = BC5602ReadRegister(CLK_CTL1_REGS);
	x |= _RST_LL_;									/* reset 1.2V register */
	BC5602WriteRegister(CLK_CTL1_REGS,x);
	Delay10us(1);
	x &= ~(_PWRON_ + _RST_LL_);				/* clear PWRON & RST_LL flag */
	BC5602WriteRegister(CLK_CTL1_REGS,x);
	
	BC5602WriteConstRegister((sizeof(BC5602RegsSetup)/2),BC5602RegsSetup);
}
/*----------------------------------------------------------------------------*/
/* @brief 	: Set crystal type.
 	@param	: xo_il - crystal oscillator low current mode.
				  xo_trim - Trim value for the internal capacitor load for the crystal.
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void BC5602CrystalSetup(u8 xo_il,u8 xo_trim)
{
	xo_cap_t reg;

	reg.value = 0;	
	reg.bits.xo_il = xo_il;
	reg.bits.xo_trim = xo_trim;
	BC5602WriteRegister(XO_CAP_CTL_REGS,reg.value);
}
/*----------------------------------------------------------------------------*/
/* @brief 	: wait crystal ready
 	@param	:
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void BC5602WaitCrystalReady(void)
{
	while(!(BC5602ReadRegister(CLK_CTL1_REGS) & _XCLK_RDY_));
	DelayXmSec(3*T1MS);
}
/*----------------------------------------------------------------------------*/
/* @brief 	: check VCO Calibration
 	@param	:
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/	
void BC5602AutoCalibration(void)
{
	u8 RegisterValue;
	
	RegisterValue = BC5602ReadRegister(OPM_CTL_REGS);
	BC5602WriteRegister(OPM_CTL_REGS,RegisterValue | _ACAL_EN_);
	while(BC5602ReadRegister(OPM_CTL_REGS) & _ACAL_EN_);
}
/*----------------------------------------------------------------------------*/
/* @brief 	: Set radio's operation mode.
 * Use this function to enter PTX (primary TX) or PRX (primary RX).
 	@param	: op_mode - operation mode select
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void BC5602SetPrimaryMode(u8 op_mode)
{
	prim_ctl_t	reg;
	
	reg.value = BC5602ReadRegister(IRQ_EN_REGS);
	reg.bits.prim_rx = (op_mode == PRIM_PRX) ? 1:0;
	BC5602WriteRegister(IRQ_EN_REGS,reg.value);
}
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Get operation mode
 * Use this function to get the operation mode.
 * @param	:
 * @return 	:	operation mode																	*/
/*----------------------------------------------------------------------------*/
u8	BC5602GetOperationMode(void)
{
	om_status_t	reg;
	
	reg.value = BC5602ReadRegister(OPM_STS_REGS);
	return(reg.bits.omst);
}
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Set the CRC mode used by the radio.
 * Use this function to set the CRC mode; CRC disabled, 1 or 2 bytes.
 	@param	: 	crc_mode - CRC mode to use
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void BC5602SetCRCMode(u8 crc_mode)
{
	crc_ctl_t reg;
	reg.value = BC5602ReadRegister(CRC_CTL_REGS);
  	switch (crc_mode)
	{
		case CRC_OFF:
			reg.bits.crc_en = 0;
			reg.bits.crc8 = 0;		
			break;
		case CRC_8BIT:
			reg.bits.crc_en = 1;
			reg.bits.crc8 = 1;
			break;
		case CRC_16BIT:
			reg.bits.crc_en = 1;
			reg.bits.crc8 = 0;
			break;
		default:
			break;
	} 
	BC5602WriteRegister(CRC_CTL_REGS,reg.value);
}
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Enable or disable interrupt for radio.
 * Use this function to enable or disable one of the interrupt sources for the radio.
 	@param	: 	int_source - Radio interrupt Source
					irq_state - Enable or Disable.
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void BC5602SetIRQMode(u8 int_source, u8 irq_state)
{
	prim_ctl_t	reg;
	reg.value = BC5602ReadRegister(IRQ_EN_REGS);
	switch(int_source)		
	{
		case IRQ_MAXRT :
			if(irq_state) 	reg.bits.mask_maxrt=0; 
			else 				reg.bits.mask_maxrt=1;
			break;
		case IRQ_TXDS :
			if(irq_state) 	reg.bits.mask_txds=0; 
			else 				reg.bits.mask_txds=1;
			break;
		case IRQ_RXDR :
			if(irq_state) 	reg.bits.mask_rxdr=0; 
			else 				reg.bits.mask_rxdr=1;
			break;
		case IRQ_ALL :
			if(irq_state)	reg.value &= ~(BIT4 | BIT5 | BIT6);
			else				reg.value |= (BIT4 | BIT5 | BIT6);
			break;
	}
	BC5602WriteRegister(IRQ_EN_REGS,reg.value);
}
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Read then clears all interrupt flags.
 * Use this function to get the interrupt flags and clear them in the same operation.
 	@param	: 	
 	@retval  :	0x10 Max Retransmit interrupt(MAXRT)									
 	@retval  :	0x20 TX Data sent interrupt(TXDS)									
 	@retval  :	0x40 RX Data received interrupt(RXDR)									*/
/*----------------------------------------------------------------------------*/
u8 BC5602GetClearIRQFlags(void)
{
	irq_status_t	reg;
	reg.value = BC5602ReadRegister(IRQ_STS_REGS);
	reg.value &= (BIT6|BIT5|BIT4);
	BC5602WriteRegister(IRQ_STS_REGS,reg.value);
	return(reg.value);
}
/*----------------------------------------------------------------------------*/
/* @brief 	: Set radio's address width.
 * Use this function to define the radio's address width,referes to both RX and TX.
 	@param	: aw - Address with in bytes
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void BC5602SetAddressWidth(u8 aw)
{
	setup_aw_t reg;	
	reg.value = BC5602ReadRegister(DMOD_CTL1_REGS);
	reg.bits.aw = aw;	
	BC5602WriteRegister(DMOD_CTL1_REGS,reg.value);
}
/*----------------------------------------------------------------------------*/
/* @brief 	: Gets the radio's address width.
 	@param	: 
 	@retval  :	Address width																	*/
/*----------------------------------------------------------------------------*/
u8 BC5602GetAddressWidth(void)
{
	setup_aw_t reg;	
	reg.value = BC5602ReadRegister(DMOD_CTL1_REGS);
	return(reg.bits.aw);	
}
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Set radio's RX address and TX address.
 * Use this function to set a RX address, or to set the TX address.
 * Beware of the difference for single and multibyte address registers.
 	@param	: 	adr - Which address to set
					adrbuf - Buffer from which the address is stored in
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void BC5602SetAddress(const u8 adr, const u8 *adrbuf)
{
	switch(adr)
	{
		case TRXADDR:
			BC5602WriteMultibyteRegister(WRITE_TRXADDR_CMD,(u8 *)adrbuf,5);
			break;
		case PIPE1:
			BC5602WriteMultibyteRegister(WRITE_RXADDR1_CMD,(u8 *)adrbuf,5);
			break;
		case PIPE2:
		case PIPE3:
		case PIPE4:
		case PIPE5:
			BC5602WriteRegister((adr-PIPE2)+RX_ADDRP2_REGS,*adrbuf);
			break;
	}
}
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Get address for selected pipe.
 * Use this function to get address for selected pipe.
 	@param	: 	adr - Which address to get
					adrbuf - buffer in which address bytes are written.
 	@retval  :	 																					*/
/*----------------------------------------------------------------------------*/
void BC5602GetAddress(const u8 adr,u8 *adrbuf)
{
	switch(adr)
	{
		case TRXADDR:
			BC5602ReadMultibyteRegister(READ_TRXADDR_CMD,adrbuf,5);
			break;
		case PIPE1:
			BC5602ReadMultibyteRegister(READ_RXADDR1_CMD,adrbuf,5);
			break;
		case PIPE2:
		case PIPE3:
		case PIPE4:
		case PIPE5:
			*adrbuf = BC5602ReadRegister((adr-PIPE2)+RX_ADDRP2_REGS);
			break;
	}
}
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Set radio's on-air datarate.
 * Use this function to select radio's on-air
 	@param	: 	datarate - On-air datarate
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void BC5602SetDataRate(u8 datarate)
{	
	setup_aw_t reg;
	
	BC5602WriteConstRegister(1,&DataRateRegsSetup[datarate]);
	reg.value = BC5602ReadRegister(DMOD_CTL1_REGS);
	reg.bits.dr = datarate;	
	BC5602WriteRegister(DMOD_CTL1_REGS,reg.value);
}
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Open radio pipe(s) and enable/ disable auto acknowledge
 * Use this function to open one or all pipes,with or without auto acknowledge.
 	@param	: 	pipe_num - Radio pipe to open
					auto_ack - Auto_Ack ON/OFF
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void BC5602OpenPipe(u8 pipe_num, u8 auto_ack)
{
	en_pipes_t en_rxaddr;
	en_pipes_t en_aa;
	
	en_rxaddr.value = BC5602ReadRegister(RX_ADDR_EN_REGS);
	en_aa.value = BC5602ReadRegister(EN_AA_REGS);
	if(pipe_num == PIPEALL)
	{
		en_rxaddr.value = 0x1F;
		en_aa.value = (auto_ack) ? 0x1F:0x00;
	}
	else
	{
		pipe_num = ((u8)1 << pipe_num);
		en_rxaddr.value |= pipe_num;
      if(auto_ack)
      {
        en_aa.value |= pipe_num;
      }
      else
      {
        en_aa.value &= ~pipe_num;
      }		
	}
	BC5602WriteRegister(RX_ADDR_EN_REGS,en_rxaddr.value);
	BC5602WriteRegister(EN_AA_REGS,en_aa.value);
}
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Close radio pipe(s).
 * Use this function to close one pipe or all pipes.
 	@param	: 	pipe_num - Pipe# number to close
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void BC5602ClosePipe(u8 pipe_num)
{
	en_pipes_t en_rxaddr;
	en_pipes_t en_aa;
	
	en_rxaddr.value = BC5602ReadRegister(RX_ADDR_EN_REGS);
	en_aa.value = BC5602ReadRegister(EN_AA_REGS);
	if(pipe_num == PIPEALL)
	{
		en_rxaddr.value = 0;
		en_aa.value = 0;
	}
	else
	{
		pipe_num = ~((u8)0x01 << pipe_num);
		en_rxaddr.value &= pipe_num;
		en_aa.value &= pipe_num;
	}
	BC5602WriteRegister(RX_ADDR_EN_REGS,en_rxaddr.value);
	BC5602WriteRegister(EN_AA_REGS,en_aa.value);
}
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Set auto acknowledge parameters.
 * Use this function to set retransmit and retransmit delay parameters.
 	@param	: 	retr - Number of retransmit, 0 equ retransmit OFF
 	@param	: 	delay - Retransmit delay in us Must be a 
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void BC5602SetAutoRetr(u8 retr, u8 delay)
{
	setup_retr_t reg;
	reg.bits.ard = delay;
	reg.bits.arc = retr;
	BC5602WriteRegister(SETUP_RETR_REGS,reg.value);
}
/*----------------------------------------------------------------------------*/
/* @brief 	: 	Set radio's TX output power.
 * Use this function set the radio's TX output power.
 	@param	: 	power - Radio's TX output power
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void BC5602SetOutputPower(u8 power)
{
	power &= 0x03;
	BC5602RegisterBank(REGS_BANK2);	
	BC5602WriteConstRegister(2,&TxPowerRegsSetup[power*2]);
//   BC5602WriteRegister(TX_PWR_CTL_REGS,TxPowerRegsSetup[power][0]);		// TX control register 2
//   BC5602WriteRegister(TX_PWR_CTL1_REGS,TxPowerRegsSetup[power][1]);   // TX control register 3
	BC5602RegisterBank(REGS_BANK0);	
}
/*----------------------------------------------------------------------------*/
/* @brief 	: 		.
 	@param	: 	
 	@retval  :																						*/
/*----------------------------------------------------------------------------*/
void BC5602TriggerCE(void)
{
	BC5602WriteRegister(CE_CTL_REGS,_CE_);
	BC5602WriteRegister(CE_CTL_REGS,~_CE_ & BIT0);	
}
/*----------------------------------------------------------------------------*/
/*	 BC5602 stroble command                                                   */
/*----------------------------------------------------------------------------*/
void BC5602StrobeCommand(u8 cmd)
{
   SPI_CSN_LOW;
	_SPIDR_ = cmd;
	SPI_delay(3);
   SPI_CSN_HIGH;
}
/*----------------------------------------------------------------------------*/
/*	 BC5602 read register                                                     */
/*----------------------------------------------------------------------------*/
u8 BC5602ReadRegister(u8 regs)
{
   regs &= REGSADDR_MASK;
   regs |= READ_REGS_CMD;
   SPI_CSN_LOW;
	_SPIDR_ = regs;
	SPI_delay(3);
	_SPIDR_ = 0xFF;
	SPI_delay(3);
   SPI_CSN_HIGH;
   return(_SPIDR_);
}
/*----------------------------------------------------------------------------*/
/*	 BC5602 write register                                                     */
/*----------------------------------------------------------------------------*/
void BC5602WriteRegister(u8 regs,u8 data)
{   
   regs &= REGSADDR_MASK;
   regs |= WRITE_REGS_CMD;
   SPI_CSN_LOW;
	_SPIDR_ = regs;
	SPI_delay(3);
	_SPIDR_ = data;
	SPI_delay(3);
   SPI_CSN_HIGH;
}
/*----------------------------------------------------------------------------*/
/*	 BC5602 read multibyte register                                            */
/*----------------------------------------------------------------------------*/
void BC5602ReadMultibyteRegister(u8 cmd,u8 *pbuf,u8 length)
{
   SPI_CSN_LOW;
	_SPIDR_ = cmd;
	_SPIMPL_ = ((u16)pbuf & 0xFF);
	_SPIMPH_ = ((u16)pbuf >> 8);
	delay_2cycle();
	delay_2cycle();	
	delay_2cycle();		
	_SPIDR_ = 0xFF;
	SPI_delay(2);
   do
   {
		delay_2cycle();
		delay_2cycle();
		asm("mov		a,%0"::"m"(_SPIDR_));
		asm("sdz		%0"::"m"(length));		
		_SPIDR_ = 0xFF;
		asm("mov		%0,a"::"m"(_SPIIAR_));		
		_SPIMPL_++;
   }while(length);
   SPI_CSN_HIGH;
}
/*----------------------------------------------------------------------------*/
/*	 BC5602 write multibyte register                                           */
/*----------------------------------------------------------------------------*/
void BC5602WriteMultibyteRegister(u8 cmd,u8 *pbuf,u8 length)
{
   SPI_CSN_LOW;
	_SPIDR_ = cmd;
	_SPIMPL_ = ((u16)pbuf & 0xFF);
	_SPIMPH_ = ((u16)pbuf >> 8);	
	delay_2cycle();
	_nop();
   do
   {
		delay_2cycle();
   	_SPIDR_ = _SPIIAR_;
		delay_2cycle();
		_SPIMPL_++;
		length--;
   }while(length);
	delay_2cycle();
	_nop();   
   SPI_CSN_HIGH;
}
/*----------------------------------------------------------------------------*/
/*	 BC5602 write multibyte register                                           */
/*----------------------------------------------------------------------------*/
void BC5602WriteConstRegister(u8 length,uc16 *pbuf)
{
	_tblp = (u16)pbuf & 0xFF;
	_tbhp = (u16)pbuf >> 8;
	asm("clr		%0.7"::"m"(_tbhp));
	GCC_RRC(_tbhp);
	GCC_RRC(_tblp);
   while(length)
   {
		asm("tabrdc	%0"::"m"(_SPIMPL_));
		asm("mov		a,%0"::"m"(_tblh));
//		_acc &= REGSADDR_MASK;		
		asm("and		a,03Fh");
		_acc |= WRITE_REGS_CMD;
   	SPI_CSN_LOW;   	
   	_SPIDR_ = _acc;
		SPI_delay(3);
   	_SPIDR_ = _SPIMPL_;
		asm("siz		tblp");
		asm("jmp		$+2");		
		asm("inc		tbhp");
		SPI_delay(2);		
   	SPI_CSN_HIGH;		
		length--;
   }
}
